function cdx_new = import_index_and_tranche_data_markit(CDX)
% --------------------------------------------------------------------------------------------------
% Import the historical CDS data in the markit format.
% --------------------------------------------------------------------------------------------------
% CDX       ... credit index structure after CDS data import from markit
% --------------------------------------------------------------------------------------------------
% sample call: import_index_and_tranche_data_markit(CDX_NA_IG2)
% --------------------------------------------------------------------------------------------------

% Import data
file_name = 'c:/thesis/data/markit/Historic Indices Tranche Data.xls';
[num, txt] = xlsread(file_name, 1, 'A1:T10000');
num_data_points = size(txt,1)-1;

% Extract and convert dates (current as well as index maturity)
dates = zeros(num_data_points, 1);
maturity_dates = zeros(num_data_points, 1);
for i=1:num_data_points
    dates(i,1) = datenum([txt{i+1,1}(4:5) '/' txt{i+1,1}(1:2) '/20' txt{i+1,1}(7:8)]);
    maturity_dates(i,1) = datenum([txt{i+1,6}(4:5) '/' txt{i+1,6}(1:2) '/20' txt{i+1,6}(7:8)]);
end

% Extract series, term, maturity of index
series = str2double(txt(2:end,3));
term = zeros(num_data_points, 1);
for i=1:num_data_points
    term(i) = str2double(txt{i+1,5}(1));
    if (term(i)==1)
        term(i) = term(i)*10 + str2double(txt{i+1,5}(2));
    end
end

% Calculate time to maturity of index (in years)
date_details = datevec(dates);
maturity_date_details = datevec(maturity_dates);
year_diff  = maturity_date_details(:,1) - date_details(:,1);
month_diff = maturity_date_details(:,2) - date_details(:,2);
day_diff   = maturity_date_details(:,3) - date_details(:,3);
T = year_diff + month_diff/12 + day_diff/12/30;

% Determine tranche depending on lower attachment point
cut_offs = [0 0.03 0.07 0.1 0.15];
lower = str2double(txt(2:end,8));
[trash, tranche_number] = ismember(lower, cut_offs);

% Extract tranche up-front, 
tranche_upfront = str2double(txt(2:end,11));
tranche_upfront(isnan(tranche_upfront)) = 0;
tranche_spread = str2double(txt(2:end,14)) * 1e4;

% Only use rows with CDX.NA.IG, standard tranche cut-offs, ...
used_rows = zeros(num_data_points, 1);
for i=1:num_data_points
    % Check if CDX.NA.IG
    name = txt{i+1,2};
    match = ~isempty(strfind(name, 'CDX.NA.IG'));
    % Check if standard cut-off
    match2 = (tranche_number(i) > 0);
    % Only use on-the-run data
    % match3 = (T(i)-term(i) >= -0.25);
    % ... only use these rows
    used_rows(i) = match && match2; % && match3;
end

% Finally, incorporate everything into the CDX structure
index = [5 7 10];
oldest_maturity = {[] [] []};
for i=1:num_data_points
    if (used_rows(i) == 0)
        continue
    end
    
    % Determine matching index for maturity
    [trash, pos] = ismember(term(i), index);
   
    % Determine position where to insert the data
    old_dates = CDX.dates{pos};
    row = find(dates(i) > [0; old_dates], 1, 'last');
    
    % Allocate memory, if necessary
    exists = ismember(dates(i), old_dates);
    if (~exists)
        % For data ...
        CDX.dates{pos} =                        [CDX.dates{pos}(1:(row-1),:); dates(i); CDX.dates{pos}(row:end,:)];
        CDX.T{pos} =                            [CDX.T{pos}(1:(row-1),:); 0; CDX.T{pos}(row:end,:)];
        CDX.index_market_price{pos} =           [CDX.index_market_price{pos}(1:(row-1),:); 0; CDX.index_market_price{pos}(row:end,:)];
        CDX.tranche_market_price_mid{pos} =     [CDX.tranche_market_price_mid{pos}(1:(row-1),:); zeros(1, 5); CDX.tranche_market_price_mid{pos}(row:end,:)];
        CDX.tranche_market_upfront_mid{pos} =   [CDX.tranche_market_upfront_mid{pos}(1:(row-1),:); zeros(1, 5); CDX.tranche_market_upfront_mid{pos}(row:end,:)];
        
        % For helper variables ...
        oldest_maturity{pos} = [oldest_maturity{pos}(1:(row-1),:); zeros(1, 5); oldest_maturity{pos}(row:end,:)];
    end
    [trash, row] = ismember(dates(i), CDX.dates{pos});
      
    % Insert data
    old_maturity = oldest_maturity{pos}(row, tranche_number(i));
    % Check if the date is older than the already inserted data (if not equal to zero)
    if ((old_maturity == 0) || (old_maturity < maturity_dates(i)))
        % Insert data
        CDX.index_market_price{pos}(row) = 0; % currently missing
        CDX.tranche_market_price_mid{pos}(row, tranche_number(i)) = tranche_spread(i);
        CDX.tranche_market_upfront_mid{pos}(row, tranche_number(i)) = tranche_upfront(i);
        CDX.T{pos}(row,1) = max(T(i), CDX.T{pos}(row,1));
        
        % Remeber inserted dat for later on
        oldest_maturity{pos}(row, tranche_number(i)) = maturity_dates(i);
    end
end
cdx_new = CDX;


